<?php
/**
 * @link http://craftcms.com/
 * @copyright Copyright (c) Pixel & Tonic, Inc.
 * @license http://craftcms.com/license
 */

namespace craft\behaviors;

use yii\base\Behavior;

/**
 * Custom field behavior
 *
 * This class provides attributes for all the unique custom field handles.
 *
 * @method $this introduction(mixed $value) Sets the [[introduction]] property
 * @method $this pagecopy(mixed $value) Sets the [[pagecopy]] property
 * @method $this pagelinks(mixed $value) Sets the [[pagelinks]] property
 * @method $this path(mixed $value) Sets the [[path]] property
 * @method $this linkname(mixed $value) Sets the [[linkname]] property
 * @method $this description(mixed $value) Sets the [[description]] property
 * @method $this serviceProviders(mixed $value) Sets the [[serviceProviders]] property
 * @method $this businessType(mixed $value) Sets the [[businessType]] property
 * @method $this contactDetails(mixed $value) Sets the [[contactDetails]] property
 * @method $this businessName(mixed $value) Sets the [[businessName]] property
 * @method $this mainImage(mixed $value) Sets the [[mainImage]] property
 */
class CustomFieldBehavior extends Behavior
{
    /**
     * @var bool Whether the behavior should provide methods based on the field handles.
     */
    public $hasMethods = false;

    /**
     * @var bool Whether properties on the class should be settable directly.
     */
    public $canSetProperties = true;

    /**
     * @var string[] List of supported field handles.
     */
    public static $fieldHandles = [
        'introduction' => true,
        'pagecopy' => true,
        'pagelinks' => true,
        'path' => true,
        'linkname' => true,
        'description' => true,
        'serviceProviders' => true,
        'businessType' => true,
        'contactDetails' => true,
        'businessName' => true,
        'mainImage' => true,
    ];

    /**
     * @var mixed Value for field with the handle “introduction”.
     */
    public $introduction;

    /**
     * @var mixed Value for field with the handle “pagecopy”.
     */
    public $pagecopy;

    /**
     * @var \craft\elements\db\MatrixBlockQuery Value for field with the handle “pagelinks”.
     */
    public $pagelinks;

    /**
     * @var string|null Value for field with the handle “path”.
     */
    public $path;

    /**
     * @var string|null Value for field with the handle “linkname”.
     */
    public $linkname;

    /**
     * @var string|null Value for field with the handle “description”.
     */
    public $description;

    /**
     * @var \craft\elements\db\MatrixBlockQuery Value for field with the handle “serviceProviders”.
     */
    public $serviceProviders;

    /**
     * @var \craft\fields\data\SingleOptionFieldData Value for field with the handle “businessType”.
     */
    public $businessType;

    /**
     * @var mixed Value for field with the handle “contactDetails”.
     */
    public $contactDetails;

    /**
     * @var string|null Value for field with the handle “businessName”.
     */
    public $businessName;

    /**
     * @var \craft\elements\db\AssetQuery Value for field with the handle “mainImage”.
     */
    public $mainImage;

    /**
     * @var array Additional custom field values we don’t know about yet.
     */
    private $_customFieldValues = [];

    /**
     * @inheritdoc
     */
    public function __call($name, $params)
    {
        if ($this->hasMethods && isset(self::$fieldHandles[$name]) && count($params) === 1) {
            $this->$name = $params[0];
            return $this->owner;
        }
        return parent::__call($name, $params);
    }

    /**
     * @inheritdoc
     */
    public function hasMethod($name)
    {
        if ($this->hasMethods && isset(self::$fieldHandles[$name])) {
            return true;
        }
        return parent::hasMethod($name);
    }

    /**
     * @inheritdoc
     */
    public function __isset($name)
    {
        if (isset(self::$fieldHandles[$name])) {
            return true;
        }
        return parent::__isset($name);
    }

    /**
     * @inheritdoc
     */
    public function __get($name)
    {
        if (isset(self::$fieldHandles[$name])) {
            return $this->_customFieldValues[$name] ?? null;
        }
        return parent::__get($name);
    }

    /**
     * @inheritdoc
     */
    public function __set($name, $value)
    {
        if (isset(self::$fieldHandles[$name])) {
            $this->_customFieldValues[$name] = $value;
            return;
        }
        parent::__set($name, $value);
    }

    /**
     * @inheritdoc
     */
    public function canGetProperty($name, $checkVars = true)
    {
        if ($checkVars && isset(self::$fieldHandles[$name])) {
            return true;
        }
        return parent::canGetProperty($name, $checkVars);
    }

    /**
     * @inheritdoc
     */
    public function canSetProperty($name, $checkVars = true)
    {
        if (!$this->canSetProperties) {
            return false;
        }
        if ($checkVars && isset(self::$fieldHandles[$name])) {
            return true;
        }
        return parent::canSetProperty($name, $checkVars);
    }
}
